/*
 * Decompiled with CFR 0.152.
 */
package club.minnced.discord.webhook;

import club.minnced.discord.webhook.IOUtil;
import club.minnced.discord.webhook.exception.HttpException;
import club.minnced.discord.webhook.receive.EntityFactory;
import club.minnced.discord.webhook.receive.ReadonlyMessage;
import club.minnced.discord.webhook.send.WebhookEmbed;
import club.minnced.discord.webhook.send.WebhookMessage;
import club.minnced.discord.webhook.send.WebhookMessageBuilder;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.jetbrains.annotations.Async;
import org.jetbrains.annotations.NotNull;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WebhookClient
implements AutoCloseable {
    public static final String WEBHOOK_URL = "https://discordapp.com/api/v7/webhooks/%s/%s?wait=%s";
    public static final String USER_AGENT = "Webhook(https://github.com/MinnDevelopment/discord-webhooks | 0.1.2)";
    private static final Logger LOG = LoggerFactory.getLogger(WebhookClient.class);
    protected final String url;
    protected final long id;
    protected final OkHttpClient client;
    protected final ScheduledExecutorService pool;
    protected final Bucket bucket;
    protected final BlockingQueue<Request> queue;
    protected final boolean parseMessage;
    protected volatile boolean isQueued;
    protected boolean isShutdown;

    protected WebhookClient(long id, String token, boolean parseMessage, OkHttpClient client, ScheduledExecutorService pool) {
        this.client = client;
        this.id = id;
        this.parseMessage = parseMessage;
        this.url = String.format(WEBHOOK_URL, Long.toUnsignedString(id), token, parseMessage);
        this.pool = pool;
        this.bucket = new Bucket();
        this.queue = new LinkedBlockingQueue<Request>();
        this.isQueued = false;
    }

    public long getId() {
        return this.id;
    }

    @NotNull
    public String getUrl() {
        return this.url;
    }

    public boolean isWait() {
        return this.parseMessage;
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull WebhookMessage message) {
        Objects.requireNonNull(message, "WebhookMessage");
        return this.execute(message.getBody());
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull File file) {
        Objects.requireNonNull(file, "File");
        return this.send(file, file.getName());
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull File file, @NotNull String fileName) {
        return this.send(new WebhookMessageBuilder().addFile(fileName, file).build());
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull byte[] data, @NotNull String fileName) {
        return this.send(new WebhookMessageBuilder().addFile(fileName, data).build());
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull InputStream data, @NotNull String fileName) {
        return this.send(new WebhookMessageBuilder().addFile(fileName, data).build());
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull WebhookEmbed first, WebhookEmbed ... embeds) {
        return this.send(WebhookMessage.embeds(first, embeds));
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull Collection<WebhookEmbed> embeds) {
        return this.send(WebhookMessage.embeds(embeds));
    }

    @NotNull
    public CompletableFuture<ReadonlyMessage> send(@NotNull String content) {
        Objects.requireNonNull(content, "Content");
        content = content.trim();
        if (content.isEmpty()) {
            throw new IllegalArgumentException("Cannot send an empty message");
        }
        if (content.length() > 2000) {
            throw new IllegalArgumentException("Content may not exceed 2000 characters");
        }
        return this.execute(WebhookClient.newBody(new JSONObject().put("content", content).toString()));
    }

    @Override
    public void close() {
        this.isShutdown = true;
        if (this.queue.isEmpty()) {
            this.pool.shutdown();
        }
    }

    protected void checkShutdown() {
        if (this.isShutdown) {
            throw new RejectedExecutionException("Cannot send to closed client!");
        }
    }

    @NotNull
    protected static RequestBody newBody(String object) {
        return RequestBody.create(IOUtil.JSON, object);
    }

    @NotNull
    protected CompletableFuture<ReadonlyMessage> execute(RequestBody body) {
        this.checkShutdown();
        return this.queueRequest(body);
    }

    @NotNull
    protected static HttpException failure(Response response) throws IOException {
        InputStream stream = IOUtil.getBody(response);
        String responseBody = stream == null ? "" : new String(IOUtil.readAllBytes(stream));
        return new HttpException("Request returned failure " + response.code() + ": " + responseBody);
    }

    @NotNull
    protected CompletableFuture<ReadonlyMessage> queueRequest(RequestBody body) {
        boolean wasQueued = this.isQueued;
        this.isQueued = true;
        CompletableFuture<ReadonlyMessage> callback = new CompletableFuture<ReadonlyMessage>();
        Request req = new Request(callback, body);
        this.enqueuePair(req);
        if (!wasQueued) {
            this.backoffQueue();
        }
        return callback;
    }

    @NotNull
    protected okhttp3.Request newRequest(RequestBody body) {
        return new Request.Builder().url(this.url).method("POST", body).header("accept-encoding", "gzip").header("user-agent", USER_AGENT).build();
    }

    protected void backoffQueue() {
        long delay = this.bucket.retryAfter();
        if (delay > 0L) {
            LOG.debug("Backing off queue for {}", (Object)delay);
        }
        this.pool.schedule(this::drainQueue, delay, TimeUnit.MILLISECONDS);
    }

    protected synchronized void drainQueue() {
        Request pair;
        boolean graceful = true;
        while (!this.queue.isEmpty() && (graceful = this.executePair(pair = (Request)this.queue.peek()))) {
        }
        boolean bl = this.isQueued = !graceful;
        if (this.isShutdown && graceful) {
            this.pool.shutdown();
        }
    }

    private boolean enqueuePair(@Async.Schedule Request pair) {
        return this.queue.add(pair);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean executePair(@Async.Execute Request req) {
        if (req.future.isCancelled()) {
            this.queue.poll();
            return true;
        }
        okhttp3.Request request = this.newRequest(req.body);
        try (Response response = this.client.newCall(request).execute();){
            this.bucket.update(response);
            if (response.code() == 429) {
                this.backoffQueue();
                boolean bl = false;
                return bl;
            }
            if (!response.isSuccessful()) {
                HttpException exception = WebhookClient.failure(response);
                LOG.error("Sending a webhook message failed with non-OK http response", exception);
                ((Request)this.queue.poll()).future.completeExceptionally(exception);
                boolean bl = true;
                return bl;
            }
            ReadonlyMessage message = null;
            if (this.parseMessage) {
                InputStream body = IOUtil.getBody(response);
                JSONObject json = IOUtil.toJSON(body);
                message = EntityFactory.makeMessage(json);
            }
            ((Request)this.queue.poll()).future.complete(message);
            if (!this.bucket.isRateLimit()) return true;
            this.backoffQueue();
            boolean bl = false;
            return bl;
        }
        catch (IOException | JSONException e) {
            LOG.error("There was some error while sending a webhook message", e);
            ((Request)this.queue.poll()).future.completeExceptionally(e);
        }
        return true;
    }

    private static final class Request {
        private final CompletableFuture<ReadonlyMessage> future;
        private final RequestBody body;

        public Request(CompletableFuture<ReadonlyMessage> future, RequestBody body) {
            this.future = future;
            this.body = body;
        }
    }

    protected static final class Bucket {
        public static final int RATE_LIMIT_CODE = 429;
        public long resetTime;
        public int remainingUses;
        public int limit = Integer.MAX_VALUE;

        protected Bucket() {
        }

        public synchronized boolean isRateLimit() {
            if (this.retryAfter() <= 0L) {
                this.remainingUses = this.limit;
            }
            return this.remainingUses <= 0;
        }

        public synchronized long retryAfter() {
            return this.resetTime - System.currentTimeMillis();
        }

        private synchronized void handleRatelimit(Response response, long current) throws IOException {
            long delay;
            String retryAfter = response.header("Retry-After");
            if (retryAfter == null) {
                InputStream stream = IOUtil.getBody(response);
                JSONObject body = IOUtil.toJSON(stream);
                delay = body.getLong("retry_after");
            } else {
                delay = Long.parseLong(retryAfter);
            }
            LOG.error("Encountered 429, retrying after {}", (Object)delay);
            this.resetTime = current + delay;
        }

        private synchronized void update0(Response response) throws IOException {
            boolean is429;
            long current = System.currentTimeMillis();
            boolean bl = is429 = response.code() == 429;
            if (is429) {
                this.handleRatelimit(response, current);
            } else if (!response.isSuccessful()) {
                LOG.debug("Failed to update buckets due to unsuccessful response with code: {} and body: \n{}", (Object)response.code(), (Object)new IOUtil.Lazy(() -> new String(IOUtil.readAllBytes(IOUtil.getBody(response)))));
                return;
            }
            this.remainingUses = Integer.parseInt(response.header("X-RateLimit-Remaining"));
            this.limit = Integer.parseInt(response.header("X-RateLimit-Limit"));
            String date = response.header("Date");
            if (date != null && !is429) {
                long reset = Long.parseLong(response.header("X-RateLimit-Reset"));
                OffsetDateTime tDate = OffsetDateTime.parse(date, DateTimeFormatter.RFC_1123_DATE_TIME);
                long delay = tDate.toInstant().until(Instant.ofEpochSecond(reset), ChronoUnit.MILLIS);
                this.resetTime = current + delay;
            }
        }

        public void update(Response response) {
            try {
                this.update0(response);
            }
            catch (Exception ex) {
                LOG.error("Could not read http response", ex);
            }
        }
    }
}

