/*
 * Copyright 2015 Austin Keener, Michael Ritter, Florian Spieß, and the JDA contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.dv8tion.jda.api.entities;

import net.dv8tion.jda.annotations.ForRemoval;
import net.dv8tion.jda.annotations.ReplaceWith;
import net.dv8tion.jda.api.JDA;
import net.dv8tion.jda.api.MessageBuilder;
import net.dv8tion.jda.api.Permission;
import net.dv8tion.jda.api.entities.emoji.CustomEmoji;
import net.dv8tion.jda.api.entities.emoji.Emoji;
import net.dv8tion.jda.api.entities.emoji.RichCustomEmoji;
import net.dv8tion.jda.api.entities.sticker.GuildSticker;
import net.dv8tion.jda.api.entities.sticker.Sticker;
import net.dv8tion.jda.api.entities.sticker.StickerItem;
import net.dv8tion.jda.api.entities.sticker.StickerSnowflake;
import net.dv8tion.jda.api.exceptions.HttpException;
import net.dv8tion.jda.api.exceptions.InsufficientPermissionException;
import net.dv8tion.jda.api.exceptions.MissingAccessException;
import net.dv8tion.jda.api.interactions.InteractionType;
import net.dv8tion.jda.api.interactions.components.ActionRow;
import net.dv8tion.jda.api.interactions.components.LayoutComponent;
import net.dv8tion.jda.api.interactions.components.buttons.Button;
import net.dv8tion.jda.api.interactions.components.selections.SelectMenu;
import net.dv8tion.jda.api.requests.RestAction;
import net.dv8tion.jda.api.requests.restaction.AuditableRestAction;
import net.dv8tion.jda.api.requests.restaction.MessageAction;
import net.dv8tion.jda.api.requests.restaction.pagination.ReactionPaginationAction;
import net.dv8tion.jda.api.utils.AttachmentOption;
import net.dv8tion.jda.api.utils.AttachmentProxy;
import net.dv8tion.jda.internal.JDAImpl;
import net.dv8tion.jda.internal.requests.FunctionalCallback;
import net.dv8tion.jda.internal.requests.Requester;
import net.dv8tion.jda.internal.utils.Checks;
import net.dv8tion.jda.internal.utils.IOUtil;
import okhttp3.OkHttpClient;
import okhttp3.Request;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.*;
import java.time.OffsetDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Represents a Text message received from Discord.
 * <br>This represents messages received from {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannels}.
 *
 * <p><b>This type is not updated. JDA does not keep track of changes to messages, it is advised to do this via events such
 * as {@link net.dv8tion.jda.api.events.message.MessageUpdateEvent MessageUpdateEvent} and similar.</b>
 *
 * <h2>Message Differences</h2>
 * There are 3 implementations of this interface in JDA.
 * <ol>
 *     <li><b>Received Message</b>
 *     <br>Messages received through events or history query.
 *         These messages hold information of <b>existing</b> messages and
 *         can be modified or deleted.</li>
 *     <li><b>System Message</b>
 *     <br>Specification of <b>Received Messages</b> that are generated by Discord
 *         on certain events. Commonly this is used in groups or to indicate a pin within a MessageChannel.
 *         The different types can be found in the {@link net.dv8tion.jda.api.entities.MessageType MessageType} enum.</li>
 *     <li><b>Data Message</b>
 *     <br>This type is produced by {@link MessageBuilder MessageBuilder}
 *         and only holds sendable information such as content or nonce. These messages do not allow
 *         any modifications via RestActions or information that is generated when sent such as the id to be used.</li>
 * </ol>
 *
 * <p>When a feature is not available it will throw an {@link java.lang.UnsupportedOperationException UnsupportedOperationException}
 * as per interface specifications.
 * <br>Specific operations may have specified information available in the {@code throws} javadoc.
 *
 * <h2>Formattable</h2>
 * This interface extends {@link java.util.Formattable Formattable} and can be used with a {@link java.util.Formatter Formatter}
 * such as used by {@link String#format(String, Object...) String.format(String, Object...)}
 * or {@link java.io.PrintStream#printf(String, Object...) PrintStream.printf(String, Object...)}.
 *
 * <p>This will use {@link #getContentDisplay()} rather than {@link Object#toString()}!
 * <br>Supported Features:
 * <ul>
 *     <li><b>Alternative</b>
 *     <br>   - Using {@link #getContentRaw()}
 *              (Example: {@code %#s} - uses {@link #getContentDisplay()})</li>
 *
 *     <li><b>Width/Left-Justification</b>
 *     <br>   - Ensures the size of a format
 *              (Example: {@code %20s} - uses at minimum 20 chars;
 *              {@code %-10s} - uses left-justified padding)</li>
 *
 *     <li><b>Precision</b>
 *     <br>   - Cuts the content to the specified size
 *              (replacing last 3 chars with {@code ...}; Example: {@code %.20s})</li>
 * </ul>
 *
 * <p>More information on formatting syntax can be found in the {@link java.util.Formatter format syntax documentation}!
 *
 * @see MessageBuilder MessageBuilder
 * @see MessageChannel#sendMessage(Message)
 *
 * @see MessageChannel#getIterableHistory()
 * @see MessageChannel#getHistory()
 * @see MessageChannel#getHistoryAfter(String, int)
 * @see MessageChannel#getHistoryBefore(String, int)
 * @see MessageChannel#getHistoryAround(String, int)
 * @see MessageChannel#getHistoryFromBeginning(int)
 * @see MessageChannel#retrieveMessageById(String)
 *
 * @see MessageChannel#deleteMessageById(String)
 * @see MessageChannel#editMessageById(String, CharSequence)
 */
public interface Message extends ISnowflake, Formattable
{
    /** Template for {@link #getJumpUrl()}.*/
    String JUMP_URL = "https://discord.com/channels/%s/%s/%s";

    /**
     * The maximum sendable file size (8 MiB)
     *
     * @see MessageAction#addFile(java.io.File, net.dv8tion.jda.api.utils.AttachmentOption...) MessageAction.addFile(...)
     */
    int MAX_FILE_SIZE = 8 << 20;

    /**
     * The maximum sendable file size for nitro (50 MiB)
     *
     * @see MessageAction#addFile(java.io.File, net.dv8tion.jda.api.utils.AttachmentOption...) MessageAction.addFile(...)
     */
    int MAX_FILE_SIZE_NITRO = 50 << 20;

    /**
     * The maximum amount of files sendable within a single message ({@value})
     *
     * @see MessageAction#addFile(java.io.File, net.dv8tion.jda.api.utils.AttachmentOption...) MessageAction.addFile(...)
     */
    int MAX_FILE_AMOUNT = 10;

    /**
     * The maximum amount of characters sendable in one message. ({@value})
     * <br>This only applies to the raw content and not embeds!
     *
     * @see MessageAction#append(CharSequence) MessageAction.append(...)
     */
    int MAX_CONTENT_LENGTH = 2000;

   /**
    * The maximum amount of reactions that can be added to one message ({@value})
    *
    * @see Message#addReaction(Emoji)
    */
    int MAX_REACTIONS = 20;

    /**
     * The maximum amount of Embeds that can be added to one message ({@value})
     *
     * @see    MessageChannel#sendMessageEmbeds(Collection)
     * @see    MessageAction#setEmbeds(Collection)
     */
    int MAX_EMBED_COUNT = 10;

    /**
     * The maximum amount of {@link Sticker Stickers} that can be added to a message ({@value})
     *
     * @see GuildMessageChannel#sendStickers(StickerSnowflake...)
     * @see MessageAction#setStickers(StickerSnowflake...)
     */
    int MAX_STICKER_COUNT = 3;

    /**
     * Pattern used to find instant invites in strings.
     *
     * <p>The only named group is at index 1 with the name {@code "code"}.
     *
     * @see #getInvites()
     */
    Pattern INVITE_PATTERN = Pattern.compile(
            "(?:https?://)?" +                     // Scheme
            "(?:\\w+\\.)?" +                       // Subdomain
            "discord(?:(?:app)?\\.com" +           // Discord domain
            "/invite|\\.gg)/(?<code>[a-z0-9-]+)" + // Path
            "(?:\\?\\S*)?(?:#\\S*)?",              // Useless query or URN appendix
            Pattern.CASE_INSENSITIVE);

    /**
     * Pattern used to find {@link #getJumpUrl() Jump URLs} in strings.
     *
     * <h4>Groups</h4>
     * <table>
     *   <caption style="display: none">Javadoc is stupid, this is not a required tag</caption>
     *   <tr>
     *     <th>Index</th>
     *     <th>Name</th>
     *     <th>Description</th>
     *   </tr>
     *   <tr>
     *     <td>0</td>
     *     <td>N/A</td>
     *     <td>The entire link</td>
     *   </tr>
     *   <tr>
     *     <td>1</td>
     *     <td>guild</td>
     *     <td>The ID of the target guild</td>
     *   </tr>
     *   <tr>
     *     <td>2</td>
     *     <td>channel</td>
     *     <td>The ID of the target channel</td>
     *   </tr>
     *   <tr>
     *     <td>3</td>
     *     <td>message</td>
     *     <td>The ID of the target message</td>
     *   </tr>
     * </table>
     * You can use the names with {@link java.util.regex.Matcher#group(String) Matcher.group(String)}
     * and the index with {@link java.util.regex.Matcher#group(int) Matcher.group(int)}.
     *
     * @see #getJumpUrl()
     */
    Pattern JUMP_URL_PATTERN = Pattern.compile(
            "(?:https?://)?" +                                             // Scheme
            "(?:\\w+\\.)?" +                                               // Subdomain
            "discord(?:app)?\\.com" +                                      // Discord domain
            "/channels/(?<guild>\\d+)/(?<channel>\\d+)/(?<message>\\d+)" + // Path
            "(?:\\?\\S*)?(?:#\\S*)?",                                      // Useless query or URN appendix
            Pattern.CASE_INSENSITIVE);

    /**
     * Returns the {@link MessageReference} for this Message. This will be null if this Message has no reference.
     *
     * <p>You can access all the information about a reference through this object.
     * Additionally, you can retrieve the referenced Message if discord did not load it in time. This can be done with {@link MessageReference#resolve()}.
     *
     * @return The message reference, or null.
     */
    @Nullable
    MessageReference getMessageReference();

    /**
     * Referenced message.
     *
     * <p>This will have different meaning depending on the {@link #getType() type} of message.
     * Usually, this is a {@link MessageType#INLINE_REPLY INLINE_REPLY} reference.
     * This can be null even if the type is {@link MessageType#INLINE_REPLY INLINE_REPLY}, when the message it references doesn't exist or discord wasn't able to resolve it in time.
     *
     * <p>This differs from a {@link MessageReference}, which contains the raw IDs attached to the reference, and allows you to retrieve the referenced message
     *
     * @return The referenced message, or null
     *
     * @see #getMessageReference()
     */
    @Nullable
    default Message getReferencedMessage()
    {
        return getMessageReference() != null
                ? getMessageReference().getMessage()
                : null;
    }

    /**
     * The {@link Mentions} used in this message.
     *
     * <p>This includes {@link Member Members}, {@link GuildChannel GuildChannels}, {@link Role Roles}, and {@link CustomEmoji CustomEmojis}.
     * Can also be used to check if a message mentions {@code @everyone} or {@code @here}.
     *
     * <p><b>Example</b><br>
     * {@code
     * System.out.println("Message mentioned these users: " + message.getMentions().getUsers());
     * System.out.println("Message used these custom emojis: " + message.getMentions().getCustomEmojis());
     * }
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return {@link Mentions} for this message.
     */
    @Nonnull
    Mentions getMentions();

    /**
     * Returns whether or not this Message has been edited before.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return True if this message has been edited.
     */
    boolean isEdited();

    /**
     * Provides the {@link java.time.OffsetDateTime OffsetDateTime} defining when this Message was last
     * edited. If this Message has not been edited ({@link #isEdited()} is {@code false}), then this method
     * will return {@code null}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Time of the most recent edit, or {@code null} if the Message has never been edited.
     */
    @Nullable
    OffsetDateTime getTimeEdited();

    /**
     * The author of this Message
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Message author
     */
    @Nonnull
    User getAuthor();

    /**
     * Returns the author of this Message as a {@link net.dv8tion.jda.api.entities.Member member}.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b> This will return {@code null}
     * if the message was not sent in a TextChannel, or if the message was sent by a Webhook.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Discord does not provide a member object for messages returned by {@link RestAction RestActions} of any kind.
     * This will return null if the message was retrieved through {@link MessageChannel#retrieveMessageById(long)} or similar means,
     * unless the member is already cached.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Message author, or {@code null} if the message was not sent in a TextChannel, or if the message was sent by a Webhook.
     *
     * @see    #isWebhookMessage()
     */
    @Nullable
    Member getMember();

    /**
     * Returns the jump-to URL for the received message. Clicking this URL in the Discord client will cause the client to
     * jump to the specified message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a data message
     *
     * @return A String representing the jump-to URL for the message
     */
    @Nonnull
    String getJumpUrl();

    /**
     * The textual content of this message in the format that would be shown to the Discord client. All
     * {@link net.dv8tion.jda.api.entities.IMentionable IMentionable} entities will be resolved to the format
     * shown by the Discord client instead of the {@literal <id>} format.
     *
     * <p>This includes resolving:
     * <br>{@link User Users} / {@link net.dv8tion.jda.api.entities.Member Members}
     * to their @Username/@Nickname format,
     * <br>{@link net.dv8tion.jda.api.entities.GuildChannel GuildChannels} to their #ChannelName format,
     * <br>{@link net.dv8tion.jda.api.entities.Role Roles} to their @RoleName format
     * <br>{@link CustomEmoji Custom Emojis} (not unicode emojis!) to their {@code :name:} format.
     *
     * <p>If you want the actual Content (mentions as {@literal <@id>}), use {@link #getContentRaw()} instead
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The textual content of the message with mentions resolved to be visually like the Discord client.
     */
    @Nonnull
    String getContentDisplay();

    /**
     * The raw textual content of this message. Does not resolve {@link net.dv8tion.jda.api.entities.IMentionable IMentionable}
     * entities like {@link #getContentDisplay()} does. This means that this is the completely raw textual content of the message
     * received from Discord and can contain mentions specified by
     * <a href="https://discord.com/developers/docs/resources/channel#message-formatting" target="_blank">Discord's Message Formatting</a>.
     *
     * @return The raw textual content of the message, containing unresolved Discord message formatting.
     */
    @Nonnull
    String getContentRaw();

    /**
     * Gets the textual content of this message using {@link #getContentDisplay()} and then strips it of markdown characters
     * like {@literal *, **, __, ~~, ||} that provide text formatting. Any characters that match these but are not being used
     * for formatting are escaped to prevent possible formatting.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The textual content from {@link #getContentDisplay()} with all text formatting characters removed or escaped.
     */
    @Nonnull
    String getContentStripped();

    /**
     * Creates an immutable List of {@link net.dv8tion.jda.api.entities.Invite Invite} codes
     * that are included in this Message.
     * <br>This will use the {@link java.util.regex.Pattern Pattern} provided
     * under {@link #INVITE_PATTERN} to construct a {@link java.util.regex.Matcher Matcher} that will
     * parse the {@link #getContentRaw()} output and include all codes it finds in a list.
     *
     * <p>You can use the codes to retrieve/validate invites via
     * {@link net.dv8tion.jda.api.entities.Invite#resolve(JDA, String) Invite.resolve(JDA, String)}
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Immutable list of invite codes
     */
    @Nonnull
    List<String> getInvites();

    /**
     * Validation <a href="https://en.wikipedia.org/wiki/Cryptographic_nonce" target="_blank" >nonce</a> for this Message
     * <br>This can be used to validate that a Message was properly sent to the Discord Service.
     * <br>To set a nonce before sending you may use {@link MessageBuilder#setNonce(String) MessageBuilder.setNonce(String)}!
     *
     * @return The validation nonce
     *
     * @since  3.4.0
     *
     * @see    MessageBuilder#setNonce(String)
     * @see    <a href="https://en.wikipedia.org/wiki/Cryptographic_nonce" target="_blank">Cryptographic Nonce - Wikipedia</a>
     */
    @Nullable
    String getNonce();

    /**
     * Used to determine if this Message was received from a {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * of the {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} specified.
     * <br>This will always be false for {@link net.dv8tion.jda.api.entities.ChannelType#VOICE} as Messages can't be sent to
     * {@link net.dv8tion.jda.api.entities.VoiceChannel VoiceChannels}.
     *
     * <p>Useful for restricting functionality to a certain type of channels.
     *
     * @param  type
     *         The {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} to check against.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return True if the {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} which this message was received
     *         from is the same as the one specified by {@code type}.
     */
    boolean isFromType(@Nonnull ChannelType type);

    /**
     * Whether this message was sent in a {@link net.dv8tion.jda.api.entities.Guild Guild}.
     * <br>If this is {@code false} then {@link #getGuild()} will throw an {@link java.lang.IllegalStateException}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return True, if {@link #getChannelType()}.{@link ChannelType#isGuild() isGuild()} is true.
     */
    default boolean isFromGuild()
    {
        return getChannelType().isGuild();
    }

    /**
     * Gets the {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} that this message was received from.
     * <br>This will never be {@link net.dv8tion.jda.api.entities.ChannelType#VOICE} as Messages can't be sent to
     * {@link net.dv8tion.jda.api.entities.VoiceChannel VoiceChannels}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The ChannelType which this message was received from.
     */
    @Nonnull
    ChannelType getChannelType();

    /**
     * Indicates if this Message was sent by a {@link net.dv8tion.jda.api.entities.Webhook Webhook} instead of a
     * {@link User User}.
     * <br>Useful if you want to ignore non-users.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return True if this message was sent by a {@link net.dv8tion.jda.api.entities.Webhook Webhook}.
     */
    boolean isWebhookMessage();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel} that this message was sent in.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The MessageChannel of this Message
     */
    @Nonnull
    MessageChannel getChannel();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.GuildMessageChannel GuildMessageChannel} that this message was sent in
     *  if it was sent in a Guild.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.Guild}.
     *
     * @return The MessageChannel of this Message
     */
    @Nonnull
    GuildMessageChannel getGuildChannel();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a PrivateChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.PrivateChannel}.
     *
     * @return The PrivateChannel this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    PrivateChannel getPrivateChannel();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.TextChannel}.
     *
     * @return The TextChannel this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    TextChannel getTextChannel();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.NewsChannel NewsChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a NewsChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.api.entities.NewsChannel NewsChannel}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.NewsChannel}.
     *
     * @return The NewsChannel this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    NewsChannel getNewsChannel();

    /**
     * The {@link net.dv8tion.jda.api.entities.Category Category} this
     * message was sent in. This will always be {@code null} for DMs.
     * <br>Equivalent to {@code getTextChannel().getParentCategory()} if this was sent in a {@link net.dv8tion.jda.api.entities.TextChannel}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return {@link net.dv8tion.jda.api.entities.Category Category} for this message
     */
    @Nullable
    Category getCategory();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.Guild Guild} that this message was sent in.
     * <br>This is just a shortcut to {@link #getTextChannel()}.{@link net.dv8tion.jda.api.entities.TextChannel#getGuild() getGuild()}.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.TextChannel}.
     *
     * @return The Guild this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    Guild getGuild();

    /**
     * An immutable list of {@link net.dv8tion.jda.api.entities.Message.Attachment Attachments} that are attached to this message.
     * <br>Most likely this will only ever be 1 {@link net.dv8tion.jda.api.entities.Message.Attachment Attachment} at most.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Immutable list of {@link net.dv8tion.jda.api.entities.Message.Attachment Attachments}.
     */
    @Nonnull
    List<Attachment> getAttachments();

    /**
     * An immutable list of {@link net.dv8tion.jda.api.entities.MessageEmbed MessageEmbeds} that are part of this
     * Message.
     *
     * @return Immutable list of all given MessageEmbeds.
     */
    @Nonnull
    List<MessageEmbed> getEmbeds();

    /**
     * Rows of interactive components such as {@link Button Buttons}.
     * <br>You can use {@link MessageAction#setActionRows(ActionRow...)} to update these.
     *
     * @return Immutable {@link List} of {@link ActionRow}
     *
     * @see    #getButtons()
     * @see    #getButtonById(String)
     */
    @Nonnull
    List<ActionRow> getActionRows();

    /**
     * All {@link Button Buttons} attached to this message.
     *
     * @return Immutable {@link List} of {@link Button Buttons}
     */
    @Nonnull
    default List<Button> getButtons()
    {
        return getActionRows().stream()
                .map(ActionRow::getButtons)
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }

    /**
     * Gets the {@link Button} with the specified ID.
     *
     * @param  id
     *         The id of the button
     *
     * @throws IllegalArgumentException
     *         If the id is null
     *
     * @return The {@link Button} or null of no button with that ID is present on this message
     */
    @Nullable
    default Button getButtonById(@Nonnull String id)
    {
        Checks.notNull(id, "Button ID");
        return getButtons().stream()
                .filter(it -> id.equals(it.getId()))
                .findFirst().orElse(null);
    }

    /**
     * All {@link Button Buttons} with the specified label attached to this message.
     *
     * @param  label
     *         The button label
     * @param  ignoreCase
     *         Whether to use {@link String#equalsIgnoreCase(String)} instead of {@link String#equals(Object)}
     *
     * @throws IllegalArgumentException
     *         If the provided label is null
     *
     * @return Immutable {@link List} of {@link Button Buttons} with the specified label
     */
    @Nonnull
    default List<Button> getButtonsByLabel(@Nonnull String label, boolean ignoreCase)
    {
        Checks.notNull(label, "Label");
        Predicate<Button> filter;
        if (ignoreCase)
            filter = b -> label.equalsIgnoreCase(b.getLabel());
        else
            filter = b -> label.equals(b.getLabel());
        return getButtons().stream()
                .filter(filter)
                .collect(Collectors.toList());
    }

    /**
     * All {@link MessageReaction MessageReactions} that are on this Message.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Immutable list of all MessageReactions on this message.
     *
     * @see    MessageReaction
     */
    @Nonnull
    List<MessageReaction> getReactions();

    /**
     * All {@link StickerItem StickerItems} that are in this Message.
     * <br>The returned StickerItems may only contain necessary information such as the sticker id, format type, name, and icon url.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Immutable list of all StickerItems in this message.
     */
    @Nonnull
    List<StickerItem> getStickers();

    /**
     * Defines whether or not this Message triggers TTS (Text-To-Speech).
     *
     * @return If this message is TTS.
     */
    boolean isTTS();

    /**
     * A {@link net.dv8tion.jda.api.entities.MessageActivity MessageActivity} that contains its type and party id.
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The activity, or {@code null} if no activity was added to the message.
     */
    @Nullable
    MessageActivity getActivity();

    /**
     * Edits this Message's content to the provided String.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account, or if
     *         {@code newContent}'s length is 0 or greater than 2000.
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessage(@Nonnull CharSequence newContent);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.api.entities.MessageEmbed MessageEmbeds}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND} in
     *         the {@link GuildChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  embeds
     *         the new embeds of the Message (up to 10)
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     * @throws IllegalArgumentException
     *         if any of the passed-in embeds is {@code null} or not {@link net.dv8tion.jda.api.entities.MessageEmbed#isSendable() sendable}.
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessageEmbeds(@Nonnull Collection<? extends MessageEmbed> embeds);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.api.entities.MessageEmbed MessageEmbeds}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND} in
     *         the {@link GuildChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  embeds
     *         the new embeds of the Message (up to 10)
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     * @throws IllegalArgumentException
     *         if any of the passed-in embeds is {@code null} or not {@link net.dv8tion.jda.api.entities.MessageEmbed#isSendable() sendable}.
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction editMessageEmbeds(@Nonnull MessageEmbed... embeds)
    {
        Checks.noneNull(embeds, "MessageEmbeds");
        return editMessageEmbeds(Arrays.asList(embeds));
    }

    /**
     * Edits this Message's content to the provided {@link LayoutComponent LayoutComponents}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     * <br>This will replace all the current {@link net.dv8tion.jda.api.interactions.components.Component Components},
     * such as {@link Button Buttons} or {@link SelectMenu SelectMenus} on this message.
     * The provided parameters are {@link LayoutComponent LayoutComponent} such as {@link ActionRow} which contain a list of components to arrange in the respective layout.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#INVALID_AUTHOR_EDIT INVALID_AUTHOR_EDIT}
     *     <br>Attempted to edit a message that was not sent by the currently logged in account.
     *         Discord does not allow editing of other users' Messages!</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed, or after {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL}
     *         was revoked in the {@link GuildChannel}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_CHANNEL UNKNOWN_CHANNEL}
     *     <br>The request was attempted after the channel was deleted.</li>
     * </ul>
     *
     * <h4>Example</h4>
     * <pre>{@code
     * List<ActionRow> rows = Arrays.asList(
     *   ActionRow.of(Button.success("prompt:accept", "Accept"), Button.danger("prompt:reject", "Reject")), // 1st row below message
     *   ActionRow.of(Button.link(url, "Help")) // 2nd row below message
     * );
     * message.editMessageComponents(rows).queue();
     * }</pre>
     *
     * @param  components
     *         Up to 5 new {@link LayoutComponent LayoutComponents} for the edited message, such as {@link ActionRow}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     *         or any of the component layouts is a custom implementation that is not supported by this interface
     * @throws IllegalArgumentException
     *         If null is provided, or more than 5 layouts are added
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated components
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessageComponents(@Nonnull Collection<? extends LayoutComponent> components);

    /**
     * Edits this Message's content to the provided {@link LayoutComponent LayoutComponents}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     * <br>This will replace all the current {@link net.dv8tion.jda.api.interactions.components.Component Components},
     * such as {@link Button Buttons} or {@link SelectMenu SelectMenus} on this message.
     * The provided parameters are {@link LayoutComponent LayoutComponent} such as {@link ActionRow} which contain a list of components to arrange in the respective layout.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#INVALID_AUTHOR_EDIT INVALID_AUTHOR_EDIT}
     *     <br>Attempted to edit a message that was not sent by the currently logged in account.
     *         Discord does not allow editing of other users' Messages!</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed, or after {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL}
     *         was revoked in the {@link GuildChannel}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_CHANNEL UNKNOWN_CHANNEL}
     *     <br>The request was attempted after the channel was deleted.</li>
     * </ul>
     *
     * <h4>Example</h4>
     * <pre>{@code
     * message.editMessageComponents(
     *   ActionRow.of(Button.success("prompt:accept", "Accept"), Button.danger("prompt:reject", "Reject")), // 1st row below message
     *   ActionRow.of(Button.link(url, "Help")) // 2nd row below message
     * ).queue();
     * }</pre>
     *
     * @param  components
     *         Up to 5 new {@link LayoutComponent LayoutComponents} for the edited message, such as {@link ActionRow}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     *         or any of the component layouts is a custom implementation that is not supported by this interface
     * @throws IllegalArgumentException
     *         If null is provided, or more than 5 layouts are added
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated components
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction editMessageComponents(@Nonnull LayoutComponent... components)
    {
        Checks.noneNull(components, "Components");
        return editMessageComponents(Arrays.asList(components));
    }

    /**
     * Edits this Message's content to the provided format.
     * <br>Shortcut for {@link MessageBuilder#appendFormat(String, Object...)}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND} in
     *         the {@link GuildChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  format
     *         Format String used to generate the Message's content via
     *         {@link MessageBuilder#appendFormat(String, Object...)} specification
     * @param  args
     *         The arguments to use in order to be converted in the format string
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     * @throws IllegalArgumentException
     *         If the provided format String is {@code null} or blank, or if
     *         the created message exceeds the 2000 character limit
     * @throws java.util.IllegalFormatException
     *         If a format string contains an illegal syntax,
     *         a format specifier that is incompatible with the given arguments,
     *         insufficient arguments given the format string, or other illegal conditions.
     *         For specification of all possible formatting errors,
     *         see the <a href="../util/Formatter.html#detail">Details</a>
     *         section of the formatter class specification.
     * @throws IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessageFormat(@Nonnull String format, @Nonnull Object... args);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.api.entities.Message Message}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND} in
     *         the {@link GuildChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or data message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalStateException
     *         <ul>
     *             <li>If the message attempting to be edited was not created by the currently logged in account</li>
     *             <li>If the message contains a MessageEmbed that is not
     *                 {@link net.dv8tion.jda.api.entities.MessageEmbed#isSendable() sendable}</li>
     *         </ul>
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessage(@Nonnull Message newContent);

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getGuildChannel().sendStickers(stickers).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link GuildMessageChannel#sendStickers(Collection)} and {@link MessageAction#reference(Message)}.
     *
     * @param  stickers
     *         The 1-3 stickers to send
     *
     * @throws MissingAccessException
     *         If the currently logged in account does not have {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} in this channel
     * @throws InsufficientPermissionException
     *         <ul>
     *           <li>If this is a {@link ThreadChannel} and the bot does not have {@link Permission#MESSAGE_SEND_IN_THREADS Permission.MESSAGE_SEND_IN_THREADS}</li>
     *           <li>If this is not a {@link ThreadChannel} and the bot does not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}</li>
     *         </ul>
     * @throws IllegalArgumentException
     *         <ul>
     *           <li>If any of the provided stickers is a {@link GuildSticker},
     *               which is either {@link GuildSticker#isAvailable() unavailable} or from a different guild.</li>
     *           <li>If the list is empty or has more than 3 stickers</li>
     *           <li>If null is provided</li>
     *         </ul>
     * @throws IllegalStateException
     *         If this message was not sent in a {@link Guild}
     *
     * @return {@link MessageAction}
     *
     * @see    Sticker#fromId(long)
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction replyStickers(@Nonnull Collection<? extends StickerSnowflake> stickers)
    {
        return getGuildChannel().sendStickers(stickers).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getGuildChannel().sendStickers(stickers).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link GuildMessageChannel#sendStickers(Collection)} and {@link MessageAction#reference(Message)}.
     *
     * @param  stickers
     *         The 1-3 stickers to send
     *
     * @throws MissingAccessException
     *         If the currently logged in account does not have {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} in this channel
     * @throws InsufficientPermissionException
     *         <ul>
     *           <li>If this is a {@link ThreadChannel} and the bot does not have {@link Permission#MESSAGE_SEND_IN_THREADS Permission.MESSAGE_SEND_IN_THREADS}</li>
     *           <li>If this is not a {@link ThreadChannel} and the bot does not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}</li>
     *         </ul>
     * @throws IllegalArgumentException
     *         <ul>
     *           <li>If any of the provided stickers is a {@link GuildSticker},
     *               which is either {@link GuildSticker#isAvailable() unavailable} or from a different guild.</li>
     *           <li>If the list is empty or has more than 3 stickers</li>
     *           <li>If null is provided</li>
     *         </ul>
     * @throws IllegalStateException
     *         If this message was not sent in a {@link Guild}
     *
     * @return {@link MessageAction}
     *
     * @see    Sticker#fromId(long)
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction replyStickers(@Nonnull StickerSnowflake... stickers)
    {
        return getGuildChannel().sendStickers(stickers).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendMessage(content).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendMessage(CharSequence)} and {@link MessageAction#reference(Message)}.
     *
     * @param  content
     *         The content of the reply message
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws java.lang.IllegalArgumentException
     *         if the provided text is null, empty or longer than 2000 characters
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction reply(@Nonnull CharSequence content)
    {
        return getChannel().sendMessage(content).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendMessageEmbeds(embeds).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendMessageEmbeds(MessageEmbed, MessageEmbed...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  embed
     *         The embed to reply with
     * @param  other
     *         Additional embeds to reply with
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided, any of the embeds are not {@link MessageEmbed#isSendable() sendable}, more than {@value Message#MAX_EMBED_COUNT} embeds are provided,
     *         or the sum of {@link MessageEmbed#getLength()} is greater than {@link MessageEmbed#EMBED_MAX_LENGTH_BOT}
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction replyEmbeds(@Nonnull MessageEmbed embed, @Nonnull MessageEmbed... other)
    {
        Checks.notNull(embed, "MessageEmbeds");
        Checks.noneNull(other, "MessageEmbeds");
        List<MessageEmbed> embeds = new ArrayList<>(1 + other.length);
        embeds.add(embed);
        Collections.addAll(embeds, other);
        return replyEmbeds(embeds);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendMessageEmbeds(embeds).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendMessageEmbeds(MessageEmbed, MessageEmbed...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  embeds
     *         The embeds to reply with
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided, any of the embeds are not {@link MessageEmbed#isSendable() sendable}, more than {@value Message#MAX_EMBED_COUNT} embeds are provided,
     *         or the sum of {@link MessageEmbed#getLength()} is greater than {@link MessageEmbed#EMBED_MAX_LENGTH_BOT}
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction replyEmbeds(@Nonnull Collection<? extends MessageEmbed> embeds)
    {
        return getChannel().sendMessageEmbeds(embeds).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendMessage(content).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendMessage(Message)} and {@link MessageAction#reference(Message)}.
     *
     * @param  content
     *         The content of the reply message
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction reply(@Nonnull Message content)
    {
        return getChannel().sendMessage(content).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendMessageFormat(content, args).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendMessageFormat(String, Object...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  format
     *         The string that should be formatted, if this is null or empty the content of the Message would be empty and cause a builder exception.
     * @param  args
     *         The arguments for your format
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction replyFormat(@Nonnull String format, @Nonnull Object... args)
    {
        return getChannel().sendMessageFormat(format, args).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendFile(file, options).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendFile(File, net.dv8tion.jda.api.utils.AttachmentOption...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  file
     *         The file to upload to the channel in the reply
     * @param  options
     *         Possible options to apply to this attachment, such as marking it as spoiler image
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction reply(@Nonnull File file, @Nonnull AttachmentOption... options)
    {
        return getChannel().sendFile(file, options).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendFile(data, name, options).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendFile(File, String, net.dv8tion.jda.api.utils.AttachmentOption...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  data
     *         The data to upload to the channel in the reply
     * @param  name
     *         The name that should be sent to discord
     * @param  options
     *         Possible options to apply to this attachment, such as marking it as spoiler image
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction reply(@Nonnull File data, @Nonnull String name, @Nonnull AttachmentOption... options)
    {
        return getChannel().sendFile(data, name, options).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendFile(data, name, options).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendFile(InputStream, String, net.dv8tion.jda.api.utils.AttachmentOption...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  data
     *         The data to upload to the channel in the reply
     * @param  name
     *         The name that should be sent to discord
     * @param  options
     *         Possible options to apply to this attachment, such as marking it as spoiler image
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction reply(@Nonnull InputStream data, @Nonnull String name, @Nonnull AttachmentOption... options)
    {
        return getChannel().sendFile(data, name, options).reference(this);
    }

    /**
     * Replies and references this message.
     * <br>This is identical to {@code message.getChannel().sendFile(data, name, options).reference(message)}.
     * You can use {@link MessageAction#mentionRepliedUser(boolean) mentionRepliedUser(false)} to not mention the author of the message.
     * <br>By default there won't be any error thrown if the referenced message does not exist.
     * This behavior can be changed with {@link MessageAction#failOnInvalidReply(boolean)}.
     *
     * <p>For further info, see {@link MessageChannel#sendFile(byte[], String, net.dv8tion.jda.api.utils.AttachmentOption...)} and {@link MessageAction#reference(Message)}.
     *
     * @param  data
     *         The data to upload to the channel in the reply
     * @param  name
     *         The name that should be sent to discord
     * @param  options
     *         Possible options to apply to this attachment, such as marking it as spoiler image
     *
     * @throws UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this is a {@link GuildChannel} and the logged in account does
     *         not have {@link Permission#MESSAGE_SEND Permission.MESSAGE_SEND}
     * @throws IllegalArgumentException
     *         If null is provided
     *
     * @return {@link MessageAction} Providing the {@link Message} created from this upload.
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    default MessageAction reply(@Nonnull byte[] data, @Nonnull String name, @Nonnull AttachmentOption... options)
    {
        return getChannel().sendFile(data, name, options).reference(this);
    }

    /**
     * Deletes this Message from Discord.
     * <br>If this Message was not sent by the currently logged in account, then this will fail unless the Message is from
     * a {@link GuildChannel} and the current account has
     * {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the channel.
     *
     * <p><u>To delete many messages at once in a {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * you should use {@link net.dv8tion.jda.api.entities.MessageChannel#purgeMessages(List) MessageChannel.purgeMessages(List)} instead.</u>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The delete was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The delete was attempted after the account lost {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link GuildChannel} when deleting another Member's message
     *         or lost {@link Permission#MESSAGE_MANAGE}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this Message was not sent by the currently logged in account, the Message was sent in a
     *         {@link GuildChannel GuildChannel}, and the currently logged in account
     *         does not have {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the channel.
     * @throws java.lang.IllegalStateException
     *         <ul>
     *              <li>If this Message was not sent by the currently logged in account and it was <b>not</b> sent in a
     *              {@link GuildChannel GuildChannel}.</li>
     *              <li>If this Message is ephemeral</li>
     *              <li>If this message type cannot be deleted. (See {@link MessageType#canDelete()})</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.api.requests.restaction.AuditableRestAction AuditableRestAction}
     *
     * @see    net.dv8tion.jda.api.entities.TextChannel#deleteMessages(java.util.Collection) TextChannel.deleteMessages(Collection)
     * @see    net.dv8tion.jda.api.entities.MessageChannel#purgeMessages(java.util.List) MessageChannel.purgeMessages(List)
     */
    @Nonnull
    @CheckReturnValue
    AuditableRestAction<Void> delete();

    /**
     * Returns the {@link net.dv8tion.jda.api.JDA JDA} instance related to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return  the corresponding JDA instance
     */
    @Nonnull
    JDA getJDA();

    /**
     * Whether or not this Message has been pinned in its parent channel.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return True - if this message has been pinned.
     */
    boolean isPinned();

    /**
     * Used to add the Message to the {@link #getChannel() MessageChannel's} pinned message list.
     * <br>This is a shortcut method to {@link MessageChannel#pinMessageById(String)}.
     *
     * <p>The success or failure of this action will not affect the return of {@link #isPinned()}.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The pin request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The pin request was attempted after the account lost {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link GuildChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this Message is from a {@link GuildChannel} and:
     *         <br><ul>
     *             <li>Missing {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL}.
     *             <br>The account needs access the the channel to pin a message in it.</li>
     *             <li>Missing {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *             <br>Required to actually pin the Message.</li>
     *         </ul>
     * @throws IllegalStateException
     *         If this Message is ephemeral
     *
     * @return {@link RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> pin();

    /**
     * Used to remove the Message from the {@link #getChannel() MessageChannel's} pinned message list.
     * <br>This is a shortcut method to {@link MessageChannel#unpinMessageById(String)}.
     *
     * <p>The success or failure of this action will not affect the return of {@link #isPinned()}.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The unpin request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The unpin request was attempted after the account lost {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link GuildChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a system message, or a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this Message is from a {@link GuildChannel} and:
     *         <br><ul>
     *             <li>Missing {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL}.
     *             <br>The account needs access the the channel to pin a message in it.</li>
     *             <li>Missing {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *             <br>Required to actually pin the Message.</li>
     *         </ul>
     * @throws IllegalStateException
     *         If this Message is ephemeral
     *
     * @return {@link RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> unpin();

    /**
     * Adds a reaction to this Message using an {@link Emoji}.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji below a message that have a counter beside them
     * showing how many users have reacted with the same emoji.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked
     *     <br>Also can happen if the account lost the {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#REACTION_BLOCKED REACTION_BLOCKED}
     *     <br>The user has blocked the currently logged in account and the reaction failed</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#TOO_MANY_REACTIONS TOO_MANY_REACTIONS}
     *     <br>The message already has too many reactions to proceed</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}
     *         or {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}
     *         in the {@link GuildChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided emoji was deleted, doesn't exist, or is not available to the currently logged-in account in this channel.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  emoji
     *         The {@link Emoji} to add as a reaction to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link GuildChannel}
     *         and the logged in account does not have
     *         <ul>
     *             <li>{@link Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}</li>
     *             <li>{@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *         </ul>
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@link Emoji} is null.</li>
     *             <li>If the provided {@link Emoji} is a custom emoji and cannot be used in the current channel.
     *                 See {@link RichCustomEmoji#canInteract(User, MessageChannel)} or {@link RichCustomEmoji#canInteract(Member)} for more information.</li>
     *         </ul>
     * @throws IllegalStateException
     *         If this message is ephemeral
     *
     * @return {@link RestAction}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> addReaction(@Nonnull Emoji emoji);

    /**
     * Removes all reactions from this Message.
     * <br>This is useful for moderator commands that wish to remove all reactions at once from a specific message.
     *
     * <p>Please note that you <b>can't</b> clear reactions if this message was sent in a {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}!
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The clear-reactions request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The clear-reactions request was attempted after the account lost {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link GuildChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link GuildChannel}
     *         and the currently logged in account does not have {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the channel.
     * @throws java.lang.IllegalStateException
     *         <ul>
     *             <li>If this message was <b>not</b> sent in a {@link net.dv8tion.jda.api.entities.Guild Guild}.</li>
     *             <li>If this message is ephemeral</li>
     *         </ul>
     *
     *
     * @return {@link RestAction}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> clearReactions();

    /**
     * Removes all reactions for the specified {@link Emoji}.
     *
     * <p>Please note that you <b>can't</b> clear reactions if this message was sent in a {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}!
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The currently logged in account lost access to the channel by either being removed from the guild
     *         or losing the {@link Permission#VIEW_CHANNEL VIEW_CHANNEL} permission</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided emoji was deleted, doesn't exist, or is not available to the currently logged-in account in this channel.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  emoji
     *         The {@link Emoji} to remove reactions for
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws InsufficientPermissionException
     *         If the currently logged in account does not have {@link Permission#MESSAGE_MANAGE} in the channel
     * @throws IllegalArgumentException
     *         If provided with null
     * @throws java.lang.IllegalStateException
     *         <ul>
     *             <li>If this message was <b>not</b> sent in a {@link net.dv8tion.jda.api.entities.Guild Guild}.</li>
     *             <li>If this message is ephemeral</li>
     *         </ul>
     *
     * @return {@link RestAction}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> clearReactions(@Nonnull Emoji emoji);

    /**
     * Removes own reaction from this Message using an {@link Emoji},
     * you can use {@link #removeReaction(Emoji, User)} to remove reactions from other users,
     * or {@link #clearReactions(Emoji)} to remove all reactions for the specified emoji.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emojis below a message that have a counter beside them
     * showing how many users have reacted with the same emoji.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked
     *     <br>Also can happen if the account lost the {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided emoji was deleted, doesn't exist, or is not available to the currently logged-in account in this channel.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  emoji
     *         The {@link Emoji} reaction to remove as a reaction from this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link GuildChannel}
     *         and the logged in account does not have {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@link Emoji} is null.</li>
     *             <li>If the provided {@link Emoji} is a custom emoji and cannot be used in the current channel.
     *                 See {@link RichCustomEmoji#canInteract(User, MessageChannel)} or {@link RichCustomEmoji#canInteract(Member)} for more information.</li>
     *         </ul>
     * @throws IllegalStateException
     *         If this is an ephemeral message
     *
     * @return {@link RestAction}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> removeReaction(@Nonnull Emoji emoji);

    /**
     * Removes a {@link User User's} reaction from this Message using an {@link Emoji}.
     *
     * <p>Please note that you <b>can't</b> remove reactions of other users if this message was sent in a {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}!
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emojis below a message that have a counter beside them
     * showing how many users have reacted with the same emoji.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked
     *     <br>Also can happen if the account lost the {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link GuildChannel} when removing the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided emoji was deleted, doesn't exist, or is not available to the currently logged-in account in this channel.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  emoji
     *         The {@link Emoji} reaction to remove as a reaction from this Message.
     * @param  user
     *         The {@link User} to remove the reaction for.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link GuildChannel}
     *         and the logged in account does not have {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}.
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@code emoji} is null.</li>
     *             <li>If the provided {@code emoji} cannot be used in the current channel.
     *                 See {@link RichCustomEmoji#canInteract(User, MessageChannel)} or {@link RichCustomEmoji#canInteract(Member)} for more information.</li>
     *             <li>If the provided user is null</li>
     *         </ul>
     * @throws java.lang.IllegalStateException
     *         <ul>
     *             <li>If this message was <b>not</b> sent in a
     *                 {@link net.dv8tion.jda.api.entities.Guild Guild}
     *                 <b>and</b> the given user is <b>not</b> the {@link SelfUser}.</li>
     *             <li>If this message is ephemeral</li>
     *         </ul>
     *
     *
     * @return {@link RestAction}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> removeReaction(@Nonnull Emoji emoji, @Nonnull User user);

    /**
     * This obtains the {@link User users} who reacted using the given {@link Emoji}.
     *
     * <p>Messages maintain a list of reactions, alongside a list of users who added them.
     *
     * <p>Using this data, we can obtain a {@link ReactionPaginationAction}
     * of the users who've reacted to this message.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The retrieve request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked
     *     <br>Also can happen if the account lost the {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided emoji was deleted, doesn't exist, or is not available to the currently logged-in account in this channel.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  emoji
     *         The {@link Emoji} to retrieve users for.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link GuildChannel} and the
     *         logged in account does not have {@link Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY} in the channel.
     * @throws java.lang.IllegalArgumentException
     *         If the provided {@link Emoji} is null.
     * @throws IllegalStateException
     *         If this Message is ephemeral
     *
     * @return The {@link ReactionPaginationAction} of the users who reacted with the provided emoji
     */
    @Nonnull
    @CheckReturnValue
    ReactionPaginationAction retrieveReactionUsers(@Nonnull Emoji emoji);

    /**
     * This obtains the {@link MessageReaction} for the given {@link Emoji} on this message.
     * <br>The reaction instance also store which users reacted with the specified emoji.
     *
     * <p>Messages store reactions by keeping a list of reaction names.
     *
     * @param  emoji
     *         The unicode or custom emoji of the reaction emoji
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws java.lang.IllegalArgumentException
     *         If the provided emoji is null
     *
     * @return The {@link MessageReaction} or null if not present.
     */
    @Nullable
    @CheckReturnValue
    MessageReaction getReaction(@Nonnull Emoji emoji);

    /**
     * Enables/Disables suppression of Embeds on this Message.
     * <br>Suppressing Embeds is equivalent to pressing the {@code X} in the top-right corner of an Embed inside the Discord client.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The clear-reactions request was attempted after the account lost access to the {@link GuildChannel}
     *         due to {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The suppress-embeds request was attempted after the account lost {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link GuildChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     * </ul>
     *
     * @param  suppressed
     *         Whether or not the embed should be suppressed
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link GuildChannel}
     *         and the currently logged in account does not have
     *         {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the channel.
     * @throws net.dv8tion.jda.api.exceptions.PermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}
     *         and the message was not sent by the currently logged in account.
     * @throws IllegalStateException
     *         If this Message is ephemeral
     * @return {@link net.dv8tion.jda.api.requests.restaction.AuditableRestAction AuditableRestAction} - Type: {@link java.lang.Void}
     *
     * @see    #isSuppressedEmbeds()
     */
    @Nonnull
    @CheckReturnValue
    AuditableRestAction<Void> suppressEmbeds(boolean suppressed);

    /**
     * Attempts to crosspost this message.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#ALREADY_CROSSPOSTED ALREADY_CROSSPOSTED}
     *     <br>The target message has already been crossposted.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The request was attempted after the account lost access to the
     *         {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed, or after {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL}
     *         was revoked in the {@link GuildChannel}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The request was attempted after the account lost
     *         {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the TextChannel.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>If the message has already been deleted. This might also be triggered for ephemeral messages.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_CHANNEL UNKNOWN_CHANNEL}
     *     <br>The request was attempted after the channel was deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     * @throws IllegalStateException
     *         <ul>
     *             <li>If the channel is not a {@link NewsChannel}.</li>
     *             <li>If the message is ephemeral.</li>
     *         </ul>
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the currently logged in account does not have
     *         {@link Permission#VIEW_CHANNEL Permission.VIEW_CHANNEL} in this channel
     *         or if this message is from another user and we don't have {@link Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *
     * @return {@link RestAction} - Type: {@link Message}
     *
     * @since  4.2.1
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Message> crosspost();

    /**
     * Whether embeds are suppressed for this message.
     * When Embeds are suppressed, they are not displayed on clients nor provided via API until un-suppressed.
     * <br>This is a shortcut method for checking if {@link #getFlags() getFlags()} contains
     * {@link net.dv8tion.jda.api.entities.Message.MessageFlag#EMBEDS_SUPPRESSED MessageFlag#EMBEDS_SUPPRESSED}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Whether or not Embeds are suppressed for this Message.
     *
     * @see    #suppressEmbeds(boolean)
     */
    boolean isSuppressedEmbeds();

    /**
     * Returns an EnumSet of all {@link Message.MessageFlag MessageFlags} present for this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Never-Null EnumSet of present {@link Message.MessageFlag MessageFlags}
     *
     * @see    Message.MessageFlag
     */
    @Nonnull
    EnumSet<MessageFlag> getFlags();

    /**
     * Returns the raw message flags of this message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The raw message flags
     *
     * @see    #getFlags()
     */
    long getFlagsRaw();

    /**
     * Whether this message is ephemeral.
     * <br>The message being ephemeral means it is only visible to the bot and the interacting user
     * <br>This is a shortcut method for checking if {@link #getFlags()} contains {@link MessageFlag#EPHEMERAL}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return Whether the message is ephemeral
     */
    boolean isEphemeral();

    /**
     * Returns a possibly {@code null} {@link net.dv8tion.jda.api.entities.ThreadChannel ThreadChannel} that was started from this message.
     * This can be {@code null} due to no ThreadChannel being started from it or the ThreadChannel later being deleted.
     *
     * @return The {@link net.dv8tion.jda.api.entities.ThreadChannel ThreadChannel} that was started from this message.
     */
    @Nullable
    ThreadChannel getStartedThread();

    /**
     * This specifies the {@link net.dv8tion.jda.api.entities.MessageType MessageType} of this Message.
     *
     * <p>Messages can represent more than just simple text sent by Users, they can also be special messages that
     * inform about events that occur. Messages can either be {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT default messages}
     * or special messages like {@link net.dv8tion.jda.api.entities.MessageType#GUILD_MEMBER_JOIN welcome messages}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The {@link net.dv8tion.jda.api.entities.MessageType MessageType} of this message.
     */
    @Nonnull
    MessageType getType();

    /**
     * This is sent on the message object when the message is a response to an {@link net.dv8tion.jda.api.interactions.Interaction Interaction} without an existing message.
     *
     * <p>This means responses to Message Components do not include this property, instead including a message reference object as components always exist on preexisting messages.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link MessageBuilder MessageBuilder})
     *
     * @return The {@link net.dv8tion.jda.api.entities.Message.Interaction Interaction} of this message.
     */
    @Nullable
    Interaction getInteraction();

    //TODO-v5: Docs
    @CheckReturnValue
    RestAction<ThreadChannel> createThreadChannel(String name);

    /**
     * Mention constants, useful for use with {@link java.util.regex.Pattern Patterns}
     */
    enum MentionType
    {
        /**
         * Represents a mention for a {@link User User}/{@link net.dv8tion.jda.api.entities.Member Member}
         * <br>The first and only group matches the id of the mention.
         */
        USER("<@!?(\\d+)>", "users"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.api.entities.Role Role}
         * <br>The first and only group matches the id of the mention.
         */
        ROLE("<@&(\\d+)>", "roles"),
        /**
         * Represents a mention for a {@link GuildChannel}
         * <br>The first and only group matches the id of the mention.
         */
        CHANNEL("<#(\\d+)>", null),
        /**
         * Represents a mention for a {@link CustomEmoji}
         * <br>The first group matches the name of the emoji and the second the id of the mention.
         */
        EMOJI("<a?:([a-zA-Z0-9_]+):([0-9]+)>", null),
        /**
         * Represents a mention for all active users, literal {@code @here}
         */
        HERE("@here", "everyone"),
        /**
         * Represents a mention for all users in a server, literal {@code @everyone}.
         */
        EVERYONE("@everyone", "everyone");

        private final Pattern pattern;
        private final String parseKey;

        MentionType(String regex, String parseKey)
        {
            this.pattern = Pattern.compile(regex);
            this.parseKey = parseKey;
        }

        @Nonnull
        public Pattern getPattern()
        {
            return pattern;
        }

        /**
         * The Key returned by this method is used to determine the group or parsable mention group they are part of.
         * <br>It is used internally in methods like {@link net.dv8tion.jda.api.requests.restaction.MessageAction#allowedMentions(Collection) MessageAction#allowedMentions(Collection)}.
         * <p>
         * Returns {@code null}, when they don't belong to any mention group.
         *
         * @return Nullable group key for mention parsing
         */
        @Nullable
        public String getParseKey()
        {
            return parseKey;
        }
    }

    /**
     * Enum representing the flags on a Message.
     * <p>
     * Note: The Values defined in this Enum are not considered final and only represent the current State of <i>known</i> Flags.
     */
    enum MessageFlag
    {
        /**
         * The Message has been published to subscribed Channels (via Channel Following)
         */
        CROSSPOSTED(0),
        /**
         * The Message originated from a Message in another Channel (via Channel Following)
         */
        IS_CROSSPOST(1),
        /**
         * Embeds are suppressed on the Message.
         * @see net.dv8tion.jda.api.entities.Message#isSuppressedEmbeds() Message#isSuppressedEmbeds()
         */
        EMBEDS_SUPPRESSED(2),
        /**
         * Indicates, that the source message of this crosspost was deleted.
         * This should only be possible in combination with {@link #IS_CROSSPOST}
         */
        SOURCE_MESSAGE_DELETED(3),
        /**
         * Indicates, that this Message came from the urgent message system
         */
        URGENT(4),
        /**
         * Indicates, that this Message is ephemeral, the Message is only visible to the bot and the interacting user
         * @see Message#isEphemeral
         */
        EPHEMERAL(6),
        /**
         * Indicates, that this Message is an interaction response and the bot is "thinking"
         */
        LOADING(7);

        private final int value;

        MessageFlag(int offset)
        {
            this.value = 1 << offset;
        }

        /**
         * Returns the value of the MessageFlag as represented in the bitfield. It is always a power of 2 (single bit)
         * @return Non-Zero bit value of the field
         */
        public int getValue()
        {
            return value;
        }

        /**
         * Given a bitfield, this function extracts all Enum values according to their bit values and returns
         * an EnumSet containing all matching MessageFlags
         * @param  bitfield
         *         Non-Negative integer representing a bitfield of MessageFlags
         * @return Never-Null EnumSet of MessageFlags being found in the bitfield
         */
        @Nonnull
        public static EnumSet<MessageFlag> fromBitField(int bitfield)
        {
            Set<MessageFlag> set = Arrays.stream(MessageFlag.values())
                .filter(e -> (e.value & bitfield) > 0)
                .collect(Collectors.toSet());
            return set.isEmpty() ? EnumSet.noneOf(MessageFlag.class) : EnumSet.copyOf(set);
        }

        /**
         * Converts a Collection of MessageFlags back to the integer representing the bitfield.
         * This is the reverse operation of {@link #fromBitField(int)}.
         * @param  coll
         *         A Non-Null Collection of MessageFlags
         * @throws IllegalArgumentException
         *         If the provided Collection is {@code null}
         * @return Integer value of the bitfield representing the given MessageFlags
         */
        public static int toBitField(@Nonnull Collection<MessageFlag> coll)
        {
            Checks.notNull(coll, "Collection");
            int flags = 0;
            for (MessageFlag messageFlag : coll)
            {
                flags |= messageFlag.value;
            }
            return flags;
        }
    }

    /**
     * Represents a {@link net.dv8tion.jda.api.entities.Message Message} file attachment.
     */
    class Attachment implements ISnowflake
    {
        private static final Set<String> IMAGE_EXTENSIONS = new HashSet<>(Arrays.asList("jpg",
                "jpeg", "png", "gif", "webp", "tiff", "svg", "apng"));
        private static final Set<String> VIDEO_EXTENSIONS = new HashSet<>(Arrays.asList("webm",
                "flv", "vob", "avi", "mov", "wmv", "amv", "mp4", "mpg", "mpeg", "gifv"));
        private final long id;
        private final String url;
        private final String proxyUrl;
        private final String fileName;
        private final String contentType;
        private final String description;
        private final int size;
        private final int height;
        private final int width;
        private final boolean ephemeral;

        private final JDAImpl jda;

        public Attachment(long id, String url, String proxyUrl, String fileName, String contentType, String description, int size, int height, int width, boolean ephemeral, JDAImpl jda)
        {
            this.id = id;
            this.url = url;
            this.proxyUrl = proxyUrl;
            this.fileName = fileName;
            this.contentType = contentType;
            this.description = description;
            this.size = size;
            this.height = height;
            this.width = width;
            this.ephemeral = ephemeral;
            this.jda = jda;
        }

        /**
         * The corresponding JDA instance for this Attachment
         *
         * @return The corresponding JDA instance for this Attachment
         */
        @Nonnull
        public JDA getJDA()
        {
            return jda;
        }

        @Override
        public long getIdLong()
        {
            return id;
        }

        /**
         * The url of the Attachment, most likely on the Discord servers.
         *
         * @return Non-null String containing the Attachment URL.
         */
        @Nonnull
        public String getUrl()
        {
            return url;
        }

        /**
         * Url to the resource proxied by the Discord CDN.
         *
         * @return Non-null String containing the proxied Attachment url.
         */
        @Nonnull
        public String getProxyUrl()
        {
            return proxyUrl;
        }

        /**
         * Returns an {@link AttachmentProxy} for this attachment.
         *
         * @return Non-null {@link AttachmentProxy} of this attachment
         *
         * @see    #getProxyUrl()
         */
        @Nonnull
        public AttachmentProxy getProxy()
        {
            return new AttachmentProxy(width > 0 && height > 0 ? proxyUrl : url);
        }

        /**
         * The file name of the Attachment when it was first uploaded.
         *
         * @return Non-null String containing the Attachment file name.
         */
        @Nonnull
        public String getFileName()
        {
            return fileName;
        }

        /**
         * The file extension of the Attachment when it was first uploaded.
         * <br>Null is returned if no characters follow the last occurrence of the '{@code .}' character
         * (or if the character is not present in {@link #getFileName()}).
         *
         * @return Non-null String containing the Attachment file extension, or null if it can't be determined.
         */
        @Nullable
        public String getFileExtension()
        {
            int index = fileName.lastIndexOf('.') + 1;
            return index == 0 || index == fileName.length() ? null : fileName.substring(index);
        }

        /**
         * The Content-Type of this file.
         * <br>This is the  <a href="https://en.wikipedia.org/wiki/Media_type" target="_blank">Media type</a> of the file that would be used in an HTTP request or similar.
         *
         * @return The content-type, or null if this isn't provided
         */
        @Nullable
        public String getContentType()
        {
            return contentType;
        }

        /**
         * The description (alt text) of this attachment.
         * <br>This description is shown when hovering over the attachment in the client.
         *
         * @return The description, or null if this isn't provided
         */
        @Nullable
        public String getDescription()
        {
            return description;
        }

        /**
         * Enqueues a request to retrieve the contents of this Attachment.
         * <br><b>The receiver is expected to close the retrieved {@link java.io.InputStream}.</b>
         *
         * <h4>Example</h4>
         * <pre>{@code
         * public void printContents(Message.Attachment attachment)
         * {
         *     attachment.retrieveInputStream().thenAccept(in -> {
         *         StringBuilder builder = new StringBuilder();
         *         byte[] buf = byte[1024];
         *         int count = 0;
         *         while ((count = in.read(buf)) > 0)
         *         {
         *             builder.append(new String(buf, 0, count));
         *         }
         *         in.close();
         *         System.out.println(builder);
         *     }).exceptionally(t -> { // handle failure
         *         t.printStackTrace();
         *         return null;
         *     });
         * }
         * }</pre>
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.InputStream}
         *
         * @deprecated Replaced by {@link #getProxy}, see {@link AttachmentProxy#download()}
         */
        @Nonnull
        @Deprecated
        @ForRemoval
        @ReplaceWith("getProxy().download()")
        public CompletableFuture<InputStream> retrieveInputStream() // it is expected that the response is closed by the callback!
        {
            CompletableFuture<InputStream> future = new CompletableFuture<>();
            Request req = getRequest();
            OkHttpClient httpClient = getJDA().getHttpClient();
            httpClient.newCall(req).enqueue(FunctionalCallback
                .onFailure((call, e) -> future.completeExceptionally(new UncheckedIOException(e)))
                .onSuccess((call, response) -> {
                    if (response.isSuccessful())
                    {
                        InputStream body = IOUtil.getBody(response);
                        if (!future.complete(body))
                            IOUtil.silentClose(response);
                    }
                    else
                    {
                        future.completeExceptionally(new HttpException(response.code() + ": " + response.message()));
                        IOUtil.silentClose(response);
                    }
                }).build());
            return future;
        }

        /**
         * Downloads the attachment into the current working directory using the file name provided by {@link #getFileName()}.
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h4>Example</h4>
         * <pre>{@code
         * public void saveLocally(Message.Attachment attachment)
         * {
         *     attachment.downloadToFile()
         *         .thenAccept(file -> System.out.println("Saved attachment to " + file.getName()))
         *         .exceptionally(t ->
         *         { // handle failure
         *             t.printStackTrace();
         *             return null;
         *         });
         * }
         * }</pre>
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.File}
         *
         * @deprecated Replaced by {@link #getProxy}, see {@link AttachmentProxy#downloadToFile(File)}
         */
        @Nonnull
        @Deprecated
        @ForRemoval
        @ReplaceWith("getProxy().downloadToFile()")
        public CompletableFuture<File> downloadToFile() // using relative path
        {
            return downloadToFile(getFileName());
        }

        /**
         * Downloads the attachment to a file at the specified path (relative or absolute).
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h4>Example</h4>
         * <pre>{@code
         * public void saveLocally(Message.Attachment attachment)
         * {
         *     attachment.downloadToFile("/tmp/" + attachment.getFileName())
         *         .thenAccept(file -> System.out.println("Saved attachment to " + file.getName()))
         *         .exceptionally(t ->
         *         { // handle failure
         *             t.printStackTrace();
         *             return null;
         *         });
         * }
         * }</pre>
         *
         * @param  path
         *         The path to save the file to
         *
         * @throws java.lang.IllegalArgumentException
         *         If the provided path is null
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.File}
         *
         * @deprecated Replaced by {@link #getProxy}, see {@link AttachmentProxy#downloadToFile(File)}
         */
        @Nonnull
        @Deprecated
        @ForRemoval
        @ReplaceWith("getProxy().downloadToFile(new File(String))")
        public CompletableFuture<File> downloadToFile(String path)
        {
            Checks.notNull(path, "Path");
            return downloadToFile(new File(path));
        }

        /**
         * Downloads the attachment to a file at the specified path (relative or absolute).
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h4>Example</h4>
         * <pre>{@code
         * public void saveLocally(Message.Attachment attachment)
         * {
         *     attachment.downloadToFile(new File("/tmp/" + attachment.getFileName()))
         *         .thenAccept(file -> System.out.println("Saved attachment to " + file.getName()))
         *         .exceptionally(t ->
         *         { // handle failure
         *             t.printStackTrace();
         *             return null;
         *         });
         * }
         * }</pre>
         *
         * @param  file
         *         The file to write to
         *
         * @throws java.lang.IllegalArgumentException
         *         If the provided file is null or cannot be written to
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.File}
         *
         * @deprecated Replaced by {@link #getProxy}, see {@link AttachmentProxy#downloadToFile(File)}
         */
        @SuppressWarnings("ResultOfMethodCallIgnored")
        @Nonnull
        @ForRemoval
        @ReplaceWith("getProxy().downloadToFile(File)")
        public CompletableFuture<File> downloadToFile(File file)
        {
            Checks.notNull(file, "File");
            try
            {
                if (!file.exists())
                    file.createNewFile();
                else
                    Checks.check(file.canWrite(), "Cannot write to file %s", file.getName());
            }
            catch (IOException e)
            {
                throw new IllegalArgumentException("Cannot create file", e);
            }

            return retrieveInputStream().thenApplyAsync((stream) -> {
                try (FileOutputStream out = new FileOutputStream(file))
                {
                    byte[] buf = new byte[1024];
                    int count;
                    while ((count = stream.read(buf)) > 0)
                    {
                        out.write(buf, 0, count);
                    }
                    return file;
                }
                catch (IOException e)
                {
                    throw new UncheckedIOException(e);
                }
                finally
                {
                    IOUtil.silentClose(stream);
                }
            }, getJDA().getCallbackPool());
        }

        /**
         * Retrieves the image of this attachment and provides an {@link net.dv8tion.jda.api.entities.Icon} equivalent.
         * <br>Useful with {@link net.dv8tion.jda.api.managers.AccountManager#setAvatar(Icon)}.
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h4>Example</h4>
         * <pre>{@code
         * public void changeAvatar(Message.Attachment attachment)
         * {
         *     attachment.retrieveAsIcon().thenCompose(icon -> {
         *         SelfUser self = attachment.getJDA().getSelfUser();
         *         AccountManager manager = self.getManager();
         *         return manager.setAvatar(icon).submit();
         *     }).exceptionally(t -> {
         *         t.printStackTrace();
         *         return null;
         *     });
         * }
         * }</pre>
         *
         * @throws java.lang.IllegalStateException
         *         If this is not an image ({@link #isImage()})
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link net.dv8tion.jda.api.entities.Icon}
         *
         * @deprecated Replaced by {@link #getProxy}, see {@link AttachmentProxy#downloadAsIcon()}
         */
        @Nonnull
        @ReplaceWith("getProxy().downloadAsIcon()")
        public CompletableFuture<Icon> retrieveAsIcon()
        {
            if (!isImage())
                throw new IllegalStateException("Cannot create an Icon out of this attachment. This is not an image.");
            return retrieveInputStream().thenApplyAsync((stream) ->
            {
                try
                {
                    return Icon.from(stream);
                }
                catch (IOException e)
                {
                    throw new UncheckedIOException(e);
                }
                finally
                {
                    IOUtil.silentClose(stream);
                }
            }, getJDA().getCallbackPool());
        }

        protected Request getRequest()
        {
            return new Request.Builder()
                .url(getUrl())
                .addHeader("user-agent", Requester.USER_AGENT)
                .addHeader("accept-encoding", "gzip, deflate")
                .build();
        }

        /**
         * The size of the attachment in bytes.
         * <br>Example: if {@code getSize()} returns 1024, then the attachment is 1024 bytes, or 1KiB, in size.
         *
         * @return Positive int containing the size of the Attachment.
         */
        public int getSize()
        {
            return size;
        }

        /**
         * The height of the Attachment if this Attachment is an image/video.
         * <br>If this Attachment is neither an image, nor a video, this returns -1.
         *
         * @return int containing image/video Attachment height, or -1 if attachment is neither image nor video.
         */
        public int getHeight()
        {
            return height;
        }

        /**
         * The width of the Attachment if this Attachment is an image/video.
         * <br>If this Attachment is neither an image, nor a video, this returns -1.
         *
         * @return int containing image/video Attachment width, or -1 if attachment is neither image nor video.
         */
        public int getWidth()
        {
            return width;
        }

        /**
         * Whether or not this attachment is from an ephemeral Message.
         * <br>If this Attachment is ephemeral, it will automatically be removed after 2 weeks. The attachment is guaranteed to be available as long as the message itself exists.
         *
         * @return True if this attachment is from an ephemeral message
         */
        public boolean isEphemeral()
        {
            return ephemeral;
        }

        /**
         * Whether or not this attachment is an Image,
         * based on {@link #getWidth()}, {@link #getHeight()}, and {@link #getFileExtension()}.
         *
         * @return True if this attachment is an image
         */
        public boolean isImage()
        {
            if (width < 0) return false; //if width is -1, so is height
            String extension = getFileExtension();
            return extension != null && IMAGE_EXTENSIONS.contains(extension.toLowerCase());
        }

        /**
         * Whether or not this attachment is a video,
         * based on {@link #getWidth()}, {@link #getHeight()}, and {@link #getFileExtension()}.
         *
         * @return True if this attachment is a video
         */
        public boolean isVideo()
        {
            if (width < 0) return false; //if width is -1, so is height
            String extension = getFileExtension();
            return extension != null && VIDEO_EXTENSIONS.contains(extension.toLowerCase());
        }

        /**
         * Whether or not this attachment is marked as spoiler,
         * based on {@link #getFileName()}.
         *
         * @return True if this attachment is marked as spoiler
         *
         * @since  4.2.1
         */
        public boolean isSpoiler()
        {
            return getFileName().startsWith("SPOILER_");
        }

    }

    /**
     * Represents an {@link net.dv8tion.jda.api.interactions.Interaction Interaction} provided with a {@link net.dv8tion.jda.api.entities.Message Message}.
     */
    class Interaction implements ISnowflake
    {
        private final long id;
        private final int type;
        private final String name;
        private final User user;
        private final Member member;

        public Interaction(long id, int type, String name, User user, Member member)
        {
            this.id = id;
            this.type = type;
            this.name = name;
            this.user = user;
            this.member = member;
        }

        @Override
        public long getIdLong()
        {
            return id;
        }

        /**
         * The raw interaction type.
         * <br>It is recommended to use {@link #getType()} instead.
         *
         * @return The raw interaction type
         */
        public int getTypeRaw()
        {
            return type;
        }

        /**
         * The {@link net.dv8tion.jda.api.interactions.InteractionType} for this interaction.
         *
         * @return The {@link net.dv8tion.jda.api.interactions.InteractionType} or {@link net.dv8tion.jda.api.interactions.InteractionType#UNKNOWN}
         */
        @Nonnull
        public InteractionType getType()
        {
            return InteractionType.fromKey(getTypeRaw());
        }

        /**
         * The command name.
         *
         * @return The command name
         */
        @Nonnull
        public String getName()
        {
            return name;
        }

        /**
         * The {@link User} who caused this interaction.
         *
         * @return The {@link User}
         */
        @Nonnull
        public User getUser()
        {
            return user;
        }

        /**
         * The {@link Member} who caused this interaction.
         * <br>This is null if the interaction is not from a guild.
         *
         * @return The {@link Member}
         */
        @Nullable
        public Member getMember()
        {
            return member;
        }
    }
}
